/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2025 i-net software GmbH, Berlin, Germany.
**/
package jdbc;

/**
 * With this database class it is possible to set the report data with an external 
 * Result Set instead of using a connection to a database created by i-net Clear Reports.
 * 
 * To do this you have to 
 * 1) create a Data Source Configuration with a user-defined driver for this database class
 * 2) set the name of the Data Source Configuration in the parameter datasource at runtime
 * 
 * Create a Data Source Configuration using the Data Source Manager:
 * - select "user-defined driver"
 * - set a name for the Data Source Configuration, e.g. "DataWithExternalResultSet"
 * - set this class "samples.jdbc.DataWithExternalResultSet" as Database Class for the Data Source Configuration
 * 
 * In the report url you have to set the name of the Data Source Configuration in the datasource parameter:
 * ...datasource=DataWithExternalResultSet...
 */

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;

import com.inet.report.Datasource;
import com.inet.report.Engine;
import com.inet.report.ReportException;
import com.inet.report.TableSource;
import com.inet.report.database.BaseDataFactory;
import com.inet.report.database.TableData;
import com.inet.report.database.TableSourceInfo;
import com.inet.report.database.fetch.DataCollector;
import com.inet.report.database.fetch.FetchTables;

/**
 * An example of a Database class which can be used as a Datasource for your report. This gives you greater flexibility
 * over the data for your report. See {@link DatabaseClassUsableInDesigner} for instructions on how to add this class as
 * a datasource from within i-net Designer.
 */
public class DataWithExternalResultSet extends BaseDataFactory {
    /**
     * Example method for creating a simple result set (the URL would need to be your own JDBC connection URL)
     * @return the result set
     * @throws SQLException if a database access error occurs
     * @throws ReflectiveOperationException if the driver class cannot be instantiated
     */
    private @Nonnull ResultSet fetchResultSet( String table ) throws SQLException, ReflectiveOperationException {
        String url = "jdbc:inetdae:<serverName>?database=<dbName>"; // your JDBC connection URL
        Connection con = DriverManager.getConnection( url, "<userName>", "<pwd>" ); // your connection info
        Statement st = con.createStatement();
        // The column name(s) need to be the same as in the report (rpt file)
        // If this result set have different column names as the report then you can use alias names with "AS".
        return st.executeQuery( "Select ... " );
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public @Nonnull Map<String, TableSourceInfo> getTableSourceInfos( @Nonnull Datasource ds, String catalog ) throws ReportException {
        HashMap<String, TableSourceInfo> map = new HashMap<>();
        map.put( "mytable", new TableSourceInfo( null, null, "mytable", TableSourceInfo.TYPE_TABLE ) );
        return map;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void fetchData( Engine engine, FetchTables fetchTables, DataCollector dataCollector ) throws ReportException {
        /* Methods to see which report or subreport is working with this engine
        engine.getReportTitle();
        engine.getPrompts();
        ...
        */

        try {
            for( TableSource tableSource : fetchTables.getTableSources() ) {
                TableData data = new TableData( fetchResultSet( tableSource.getDatabaseIdentifier() ) );
                dataCollector.addUnjoinedData( tableSource, data );
            }
        } catch( Exception e ) {
            e.printStackTrace();
            //...
        }
    }
}
